<?php
define('PAGE_TITLE', 'Items Management');
require_once '../config/config.php';
require_once '../libraries/Auth.php';

$auth = new Auth();
$auth->requireLogin();
$auth->requirePermission('inventory');

include '../views/layouts/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1 class="h3 mb-0">Items Management</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>inventory/">Inventory</a></li>
                <li class="breadcrumb-item active">Items</li>
            </ol>
        </nav>
    </div>
</div>

<!-- Filters -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Filters</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <label class="form-label">Category</label>
                        <select class="form-select" id="filterCategory">
                            <option value="">All Categories</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Status</label>
                        <select class="form-select" id="filterStatus">
                            <option value="">All Status</option>
                            <option value="1">Active</option>
                            <option value="0">Inactive</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Search</label>
                        <input type="text" class="form-control" id="filterSearch" placeholder="Search items...">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">&nbsp;</label>
                        <div>
                            <button class="btn btn-primary" onclick="applyFilters()">
                                <i class="fas fa-search"></i> Apply
                            </button>
                            <button class="btn btn-secondary" onclick="clearFilters()">
                                <i class="fas fa-undo"></i> Clear
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Items Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Items List</h5>
                <div>
                    <button class="btn btn-success" onclick="exportItems()">
                        <i class="fas fa-download me-2"></i>Export
                    </button>
                    <button class="btn btn-primary" onclick="openModal('addItem')">
                        <i class="fas fa-plus me-2"></i>Add Item
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover" id="itemsTable">
                        <thead>
                            <tr>
                                <th>Code</th>
                                <th>Name</th>
                                <th>Category</th>
                                <th>Stock</th>
                                <th>Avg Cost</th>
                                <th>Total Value</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be loaded via AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Item Modal -->
<div class="modal fade" id="itemModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="itemModalTitle">Add Item</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="itemForm">
                    <input type="hidden" id="itemId" name="id">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Item Code *</label>
                                <input type="text" class="form-control" id="itemCode" name="code" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Item Name *</label>
                                <input type="text" class="form-control" id="itemName" name="name" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Category *</label>
                                <select class="form-select" id="itemCategory" name="category_id" required>
                                    <option value="">Select Category</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Unit of Measure</label>
                                <input type="text" class="form-control" id="itemUnit" name="unit_of_measure" placeholder="e.g., pcs, kg, liter">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Cost Method *</label>
                                <select class="form-select" id="itemCostMethod" name="cost_method" required>
                                    <option value="FIFO">FIFO</option>
                                    <option value="AVERAGE">Average Cost</option>
                                    <option value="LIFO">LIFO</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Status</label>
                                <select class="form-select" id="itemStatus" name="is_active">
                                    <option value="1">Active</option>
                                    <option value="0">Inactive</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Minimum Stock</label>
                                <input type="number" class="form-control" id="itemMinStock" name="min_stock" min="0" value="0">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Maximum Stock</label>
                                <input type="number" class="form-control" id="itemMaxStock" name="max_stock" min="0" value="0">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea class="form-control" id="itemDescription" name="description" rows="3"></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveItem()">Save Item</button>
            </div>
        </div>
    </div>
</div>

<script>
// Load form data
function loadFormData() {
    // Load categories
    fetch('../api/inventory/categories.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const categorySelect = document.getElementById('itemCategory');
                const filterCategory = document.getElementById('filterCategory');
                
                categorySelect.innerHTML = '<option value="">Select Category</option>';
                filterCategory.innerHTML = '<option value="">All Categories</option>';
                
                data.categories.forEach(category => {
                    const option = document.createElement('option');
                    option.value = category.id;
                    option.textContent = category.name;
                    
                    categorySelect.appendChild(option.cloneNode(true));
                    filterCategory.appendChild(option.cloneNode(true));
                });
                
                initSelect2('#itemCategory');
                initSelect2('#filterCategory');
            }
        });
}

// Load items data
function loadItems() {
    const table = initDataTable('itemsTable', {
        ajax: {
            url: '../api/inventory/items-list.php',
            dataSrc: 'items'
        },
        columns: [
            { data: 'code' },
            { data: 'name' },
            { data: 'category_name' },
            { 
                data: 'total_stock',
                render: function(data, type, row) {
                    const minStock = parseFloat(row.min_stock) || 0;
                    const stock = parseFloat(data) || 0;
                    const status = stock <= minStock ? 'danger' : 'success';
                    return `<span class="badge bg-${status}">${stock}</span>`;
                }
            },
            { 
                data: 'average_cost',
                render: function(data) {
                    return formatCurrency(data || 0);
                }
            },
            { 
                data: 'total_value',
                render: function(data) {
                    return formatCurrency(data || 0);
                }
            },
            {
                data: 'is_active',
                render: function(data) {
                    return data == 1 ? 
                        '<span class="badge bg-success">Active</span>' : 
                        '<span class="badge bg-secondary">Inactive</span>';
                }
            },
            {
                data: null,
                render: function(data) {
                    return `
                        <button class="btn btn-sm btn-info" onclick="viewStock(${data.id})">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button class="btn btn-sm btn-primary" onclick="editItem(${data.id})">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-danger" onclick="deleteItem(${data.id})">
                            <i class="fas fa-trash"></i>
                        </button>
                    `;
                }
            }
        ]
    });
}

// Apply filters
function applyFilters() {
    const table = $('#itemsTable').DataTable();
    const category = $('#filterCategory').val();
    const status = $('#filterStatus').val();
    const search = $('#filterSearch').val();
    
    table.ajax.url(`../api/inventory/items-list.php?category=${category}&status=${status}&search=${search}`).load();
}

// Clear filters
function clearFilters() {
    $('#filterCategory').val('');
    $('#filterStatus').val('');
    $('#filterSearch').val('');
    
    const table = $('#itemsTable').DataTable();
    table.ajax.url('../api/inventory/items-list.php').load();
}

// Open modal for adding/editing item
function openModal(type, itemId = null) {
    const modal = new bootstrap.Modal(document.getElementById('itemModal'));
    
    if (type === 'addItem') {
        document.getElementById('itemModalTitle').textContent = 'Add Item';
        document.getElementById('itemForm').reset();
        document.getElementById('itemId').value = '';
    } else if (type === 'editItem') {
        document.getElementById('itemModalTitle').textContent = 'Edit Item';
        loadItemData(itemId);
    }
    
    modal.show();
}

// Load item data for editing
function loadItemData(itemId) {
    fetch(`../api/inventory/get-item.php?id=${itemId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const item = data.item;
                document.getElementById('itemId').value = item.id;
                document.getElementById('itemCode').value = item.code;
                document.getElementById('itemName').value = item.name;
                document.getElementById('itemCategory').value = item.category_id || '';
                document.getElementById('itemUnit').value = item.unit_of_measure || '';
                document.getElementById('itemCostMethod').value = item.cost_method || 'FIFO';
                document.getElementById('itemStatus').value = item.is_active;
                document.getElementById('itemMinStock').value = item.min_stock || 0;
                document.getElementById('itemMaxStock').value = item.max_stock || 0;
                document.getElementById('itemDescription').value = item.description || '';
            }
        });
}

// Save item
function saveItem() {
    const form = document.getElementById('itemForm');
    const formData = new FormData(form);
    
    fetch('../api/inventory/save-item.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccess('Item saved successfully!');
            $('#itemModal').modal('hide');
            $('#itemsTable').DataTable().ajax.reload();
        } else {
            showError(data.message || 'Failed to save item');
        }
    });
}

// Delete item
function deleteItem(itemId) {
    confirmDialog('Delete Item', 'Are you sure you want to delete this item?', function() {
        fetch('../api/inventory/delete-item.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ id: itemId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showSuccess('Item deleted successfully!');
                $('#itemsTable').DataTable().ajax.reload();
            } else {
                showError(data.message || 'Failed to delete item');
            }
        });
    });
}

// Export items
function exportItems() {
    window.location.href = '../api/inventory/export-items.php';
}

// View stock details
function viewStock(itemId) {
    window.location.href = `stock-detail.php?item=${itemId}`;
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    loadFormData();
    loadItems();
    
    // Filter event listeners
    document.getElementById('filterCategory').addEventListener('change', applyFilters);
    document.getElementById('filterStatus').addEventListener('change', applyFilters);
    document.getElementById('filterSearch').addEventListener('input', applyFilters);
});
</script>

<?php include '../views/layouts/footer.php'; ?>